/* PC speaker driver. Accepts 8 bit unsigned 20k mono audio data */
#include <SupportDefs.h>
#include <KernelExport.h>
#include <Drivers.h>
#include <ISA.h>

/*****************************************************************************************************************
 This defines the maximum interrupt latency in microseconds.  We generally always want to interrupt early and spin
 until the proper time.  In general, the actual overhead for an interrupt is small, however, if other drivers have
 interrupts disabled or are servicing interrupts, the actual latency could increase.  MIN_INTERRUPT_INTERVAL +
 INTERRUPT_LATENCY must be less than the OUTPUT_SAMPLE_PERIOD, Otherwise your machine will hang. */
 
#define INTERRUPT_LATENCY 10
#define MAX_PLAY_BUFFER_LENGTH 0x4000

/* Used to load and unload driver */
status_t init_hardware (void);
status_t init_driver(void);
void uninit_driver(void);
/* Communication to/from driver */
static status_t speaker_open(const char *name, uint32 flags, void **cookie);
static status_t speaker_close(void *cookie);
static status_t speaker_free(void *cookie);
static status_t speaker_control(void *cookie, uint32 op, void *buf, size_t len);
static status_t speaker_read(void *cookie, off_t pos, void *buf, size_t *len);
static status_t speaker_write(void *cookie, off_t pos, const void *buf, size_t *len);
static status_t play_chunk(char *buffer, long len);
/* Internal Data */
const char **publish_devices(void);
device_hooks* find_device(const char *name);

static int32 speaker_interrupt(timer *_t, uint32 flags);
 
typedef struct {
 struct    timer timer;
// struct timer = { qent entry; uint16 flags; uint16 cpu; timer_hook hook; bigtime_t  period; }
 long      buffer_size;
 long      current_sample;
 bigtime_t frame_length;
 bigtime_t next_frame;
 char     *buffer;
} speaker_timer_info;

const char *speaker_names[] = { "audio/pcspeaker",NULL };

/* Device Hooks - The hook functions specified in the device_hooks function returned by the driver's find_device()
   function handle requests made by devfs (and through devfs, from user applications). These are described in this
   section.  The structure itself looks like this: 
   typedef struct { 
           device_open_hook open;       -> open entry point
           device_close_hook close;     -> close entry point
           device_free_hook free;       -> free cookie
           device_control_hook control; -> control entry point
           device_read_hook read;       -> read entry point
           device_write_hook write;     -> write entry point
           device_select_hook select;   -> select entry point
           device_deselect_hook deselect; -> deselect entry point
           device_readv_hook readv;     -> posix read entry point
           device_writev_hook writev;   -> posix write entry point
   } device_hooks;
   In all cases, return B_OK if the operation is successfully completed, or an appropriate error code if not.
   function pointers for the device hooks entry points */

static device_hooks speaker_device = {
 speaker_open, speaker_close, speaker_free, speaker_control, speaker_read, speaker_write,
 NULL, NULL, NULL, NULL }; /* speaker_select, speaker_deselect, speaker_readv, speaker_writev */

static speaker_timer_info speaker_timer;
static int32 open_count = 0;
static isa_module_info *isa = NULL;
static sem_id done_sem = -1;
static bool is_speaker_on = false;
static int32 frame_rate;
static sem_id write_lock = -1;
static char play_buffer[MAX_PLAY_BUFFER_LENGTH];
static long play_buffer_length = 0;
static bigtime_t next_cone_movement;

/*****************************************************************************************************************
 init_hardware - status_t init_hardware (void) - This function is called when the system is booted, which lets the
 driver detect and reset the hardware it controls. The function should return B_OK if the initialization is
 successful; otherwise, an appropriate error code should be returned. If this function returns an error, the
 driver won't be used. */

status_t init_hardware (void) {
 return B_OK; } 

/*****************************************************************************************************************
 init_driver - status_t init_driver(void) - optional function - called every time the driver is loaded.  Drivers
 are loaded and unloaded on an as-needed basis. When a driver is loaded by devfs, this function is called to let
 the driver allocate memory and other needed system resources. Return B_OK if initialization succeeds, otherwise
 return an appropriate error code. <<<what happens if this returns an error?>>> */

status_t init_driver(void) {
 status_t ret = B_OK; ret = get_module(B_ISA_MODULE_NAME, (module_info**) &isa); if (ret != B_OK) { goto error1; }
 done_sem   = create_sem(0, "finished");   if (done_sem < B_OK)   { ret = done_sem;   goto error2; }
 write_lock = create_sem(1, "write lock"); if (write_lock < B_OK) { ret = write_lock; goto error3; }
 is_speaker_on = false; frame_rate = 16000; return B_OK;
error3:
 delete_sem(done_sem);
error2:
 put_module(B_ISA_MODULE_NAME);
error1:
 return ret; }

/*****************************************************************************************************************
 uninit_driver - void uninit_driver(void) - optional function - called every time the driver is unloaded.  This
 function is called by devfs just before the driver is unloaded from memory. This lets the driver clean up after
 itself, freeing any resources it allocated. */

void uninit_driver(void) {
 return; }

/*****************************************************************************************************************
 open_hook() - status_t open_hook(const char *name, uint32 flags, void **cookie) - This hook function is called
 when a program opens one of the devices supported by the driver. The name of the device (as returned by
 publish_devices()) is passed in name, along with the flags passed to the Posix open() function. cookie points to
 space large enough for you to store a single pointer. You can use this to store state information specific to the
 open() instance. If you need to track information on a per-open() basis, allocate the memory you need and store a
 pointer to it in *cookie. */

static status_t speaker_open(const char *name, uint32 flags, void **cookie) {
 if (atomic_add(&open_count, 1) < 1) {
    *cookie = NULL; return B_OK;
 } else {
    atomic_add(&open_count, -1); return B_ERROR; } }

/*****************************************************************************************************************
 close_hook() - status_t close_hook(void *cookie) - This hook is called when an open instance of the driver is
 closed using the close() Posix function. Note that because of the multithreaded nature of the BeOS, it's possible
 there may still be transactions pending, and you may receive more calls on the device. For that reason, you
 shouldn't free instance-wide system resources here. Instead, you should do this in free_hook(). However, if there
 are any blocked transactions pending, you should unblock them here. */

static status_t speaker_close(void *cookie) {
 atomic_add(&open_count, -1); return B_OK; }

/*****************************************************************************************************************
 free_hook() - status_t free_hook(void *cookie) - This hook is called once all pending transactions on an open
 (but closing) instance of your driver are completed. This is where your driver should release instance wide
 system resources. free_hook() doesn't correspond to any Posix function. */

static status_t speaker_free(void *cookie) {
 delete_sem(write_lock); delete_sem(done_sem); put_module(B_ISA_MODULE_NAME); return B_OK; }

/*****************************************************************************************************************
 control_hook() - status_t control_hook(void *cookie, uint32 op, void *data, size_t len) - This hook handles the
 ioctl() function for an open instance of your driver. The control hook provides a means to perform operations
 that don't map directly to either read() or write(). It receives the cookie for the open instance, plus the
 command code op and the data and len arguments specified by ioctl()'s caller.  These arguments have no inherent
 relationship; they're simply arguments to ioctl() that are forwarded to your hook function. Their definitions are
 defined by the driver. Common command codes can be found in be/drivers/Drivers.h.  The len argument is only valid
 when ioctl() is called from user space; the kernel always sets it to 0. */

static status_t speaker_control(void *cookie, uint32 op, void *buf, size_t len) {
 return B_ERROR; }

/*****************************************************************************************************************
 read_hook() - status_t read_hook(void *cookie, off_t position, void *data, size_t *len) - This hook handles the
 Posix read() function for an open instance of your driver.  Implement it to read len bytes of data starting at
 the specified byte position on the device, storing the read bytes at data. Exactly what this does is
 device-specific (disk devices would read from the specified offset on the disk, but a graphics driver might have
 some other interpretation of this request). Before returning, you should set len to the actual number of bytes
 read into the buffer. Return B_OK if data was read (even if the number of returned bytes is less than requested),
 otherwise return an appropriate error. */

static status_t speaker_read(void *cookie, off_t pos, void *buf, size_t *len) {
 return B_ERROR; }

/****************************************************************************************************************/

static int32 speaker_interrupt(timer *_t, uint32 flags) {
 speaker_timer_info *t = (speaker_timer_info*) _t;
 while (system_time() < t->next_frame) { } // Wait till time to play next sample
 isa->write_io_8(0x378, t->buffer[t->current_sample]);   // Write data to parallel port.
 t->current_sample++; // Point to next sample.

 if (t->current_sample >= t->buffer_size) {
    release_sem_etc(done_sem, 1, B_DO_NOT_RESCHEDULE); 
    return B_INVOKE_SCHEDULER; } /* Check to see if we're done playing this buffer */

 t->next_frame = t->next_frame + t->frame_length; // Time next sample.
 add_timer((timer *)&speaker_timer, speaker_interrupt, t->next_frame - INTERRUPT_LATENCY, B_ONE_SHOT_ABSOLUTE_TIMER);/* Program an interrupt for the next cone movement. Note that we program to occur a little before the deadline.*/
 return B_HANDLED_INTERRUPT; }

/*****************************************************************************************************************
 Play the sound in the background. Set up a timer for the first interrupt, then wait on a completion semaphore.*/
// static int32 speaker_interrupt(timer *_t, uint32 flags);
// typedef struct {
//  struct    timer timer;
   // struct timer = { qent entry; uint16 flags; uint16 cpu; timer_hook hook; bigtime_t  period; }
//  long      buffer_size;
//  long      current_sample;
//  bigtime_t frame_length;
//  bigtime_t next_frame;
//  char     *buffer;
// } speaker_timer_info;

static status_t play_chunk(char *buffer, long len) {
 status_t ret;
 bigtime_t frame_length = 1000000/frame_rate;
 bigtime_t next_frame_time = (1 + (system_time() / frame_length)) * frame_length;
 speaker_timer.buffer_size = len;
 speaker_timer.current_sample = 0; 
 speaker_timer.frame_length = frame_length;
 speaker_timer.next_frame = next_frame_time;
 speaker_timer.buffer = buffer;
 
 ret = add_timer((timer*) &speaker_timer, speaker_interrupt, next_frame_time - INTERRUPT_LATENCY, B_ONE_SHOT_ABSOLUTE_TIMER);
 if (ret < 0) { return ret; }
 acquire_sem(done_sem); return B_OK; }

/*****************************************************************************************************************
 write_hook() - status_t write_hook(void *cookie, off_t position, void *data, size_t len) -  This hook handles the
 Posix write() function for an open instance of your driver. Implement it to write len bytes of data starting at
 the specified byte position on the device, from the buffer pointed to by data. Exactly what this does is
 device-specific (disk devices would write to the specified offset on the disk, but a graphics driver might have
 some other interpretation of this request). Return B_OK if data was read (even if the number of returned bytes is
 less than requested), otherwise return an appropriate error. */ 

static status_t speaker_write(void *cookie, off_t pos, const void *buffer, size_t *lenght) {
 int index; acquire_sem(write_lock); // Block writes to buffer before it is played out.
 index = 0; play_buffer_length = *lenght;
 while (play_buffer_length > 0) { // Check if buffer contains any sound data
    if (play_buffer_length > MAX_PLAY_BUFFER_LENGTH) {
       play_buffer_length = MAX_PLAY_BUFFER_LENGTH; } // If more data than size of play_buffer just fill play_buffer.
    /* We must copy the chunk from the user buffer to our own buffer in kernel space, reading from the user buffer
    could page fault, which is bad to do. */
    memcpy(play_buffer, (char*) buffer + index, play_buffer_length); // Copy buffer into play_buffer
    if (play_chunk(play_buffer, play_buffer_length) < B_OK) {
       release_sem(write_lock); return B_OK; } // Allow new writes to buffer.
    index = index + play_buffer_length; play_buffer_length = *lenght - index; }
 release_sem(write_lock); return B_OK; }

/*****************************************************************************************************************
 publish_devices - const char** publish_devices(void) - returns a null-terminated array of devices supported by
 this driver.  Devfs calls publish_devices() to learn the names, relative to /dev, of the devices the driver
 supports. The driver should return a NULL-terminated array of strings indicating all the installed devices the
 driver supports. For example, an ethernet device driver might return: 
 static char *devices[] = { "net/ether", NULL };

 In this case, devfs will then create the pseudo-file /dev/net/ether, through which all user applications can
 access the driver.  Since only one instance of the driver will be loaded, if support for multiple devices of the
 same type is desired, the driver must be capable of supporting them. If the driver senses (and supports) two
 ethernet cards, it might return: 
 static char *devices[] = { "net/ether1", "net/ether2", NULL }; */

const char** publish_devices(void) {
 return speaker_names; }

/*****************************************************************************************************************
 find_device - device_hooks* find_device(const char *name) - returns a pointer to device hooks structure for a
 given device name.  When a device published by the driver is accessed, devfs communicates with it through a
 series of hook functions that handle the requests.  The find_device() function is called to obtain a list of
 these hook functions, so that devfs can call them.  The device_hooks structure returned lists out the hook
 functions.  The device_hooks structure, and what each hook does, is described in the next section. */

device_hooks* find_device(const char *name) {
 return &speaker_device; }

/* api_version - This variable defines the API version to which the driver was written, and should be set to
 B_CUR_DRIVER_API_VERSION at compile time. The value of this variable will be changed with every revision to the
 driver API; the value with which your driver was compiled will tell devfs how it can communicate with the driver.
*/

int32 api_version = B_CUR_DRIVER_API_VERSION;

/* readv_hook() - status_t readv_hook(void *cookie, off_t position, const struct iovec *vec, size_t count, size_t *len) 
This hook handles the Posix readv() function for an open instance of your driver.  This is a scatter/gather read
function; given an array of iovec structures describing address/length pairs for a group of destination buffers,
your implementation should fill each successive buffer with bytes, up to a total of len bytes. The vec array has
count items in it. As with read_hook(), set len to the actual number of bytes read, and return an appropriate
result code.

static status_t my_device_readv(void *cookie, off_t position, const iovec *vec, size_t count, size_t *len) {
 return B_OK; }

/* writev_hook() - status_t writev_hook(void *cookie, off_t position, const struct iovec *vec, size_t count, size_t *len) 
 This hook handles the Posix writev() function for an open instance of your driver. This is a scatter/gather write
 function; given an array of iovec structures describing address/length pairs for a group of source buffers, your
 implementation should write each successive buffer to disk, up to a total of len bytes. The vec array has count
 items in it. Before returning, set len to the actual number of bytes written, and return an appropriate result
 code.

static status_t my_device_writev(void *cookie, off_t position, const iovec *vec, size_t count, size_t *len) {
  return B_OK; }

select_hook() , deselect_hook() 
 These hooks are reserved for future use. Set the corresponding entries in your device_hooks structure to NULL.

status_t my_device_select(void *cookie, uint8 event, uint32 ref, selectsync *sync) {
 return B_OK; }

status_t my_device_deselect(void *cookie, uint8 event, selectsync *sync) {
 return B_OK; } */




